#!/usr/bin/python3
# -*- coding: utf-8 -*-

#+=============================================================================================
#!
#!		A weekly based timer for the RevPi Core
#| 
#+=============================================================================================
#|	
#|		File-ID:		$Id: RevPiTimer.py 779 2017-06-02 15:01:29Z dconzelmann $
#|		Date:			$Date: 2017-06-02 17:01:29 +0200 (Fr, 02 Jun 2017) $
#|		File Revision:	$Rev: 779 $
#|	
#|		Author:			$Author: dconzelmann $
#|		Company:		$Cpn: KUNBUS GmbH $
#|
#+=============================================================================================

import sys     
import time      # used for the delays of the demo
import struct    # used for the processing of byte-strings 
import fcntl     # used for the byte access of the process image within IOCTL
import datetime
import argparse
import syslog

bVerbose    = False
bDebug      = False
bSyslog     = False

TIMER_FMT   = 'BBBBBB'
TIMER_SIZE  = struct.calcsize(TIMER_FMT)
TIMER_CNT   = 16		
STATUS_FMT  = 'H'
STATUS_SIZE = struct.calcsize(STATUS_FMT)

MODULETYPE_TIMER  = 0x7001
size_SDeviceInfo  = 72			# size of struct SDeviceInfo
MAX_MODULES       = 255			# max. amount of pictory modules
KB_GET_DEVICE_INFO_LIST = 13	# ioctl function call

dOffsets_SDeviceInfo = {
        'i8uAddress'        : 0,
        'i32uSerialnumber'  : 4,
        'i16uModuleType'    : 8,
        'i16uHW_Revision'   : 10,
        'i16uSW_Major'      : 12,
        'i16uSW_Minor'      : 14,
        'i32uSVN_Revision'  : 16,
        'i16uInputLength'   : 20,
        'i16uOutputLength'  : 22,
        'i16uConfigLength'  : 24,
        'i16uBaseOffset'    : 26,
        'i16uInputOffset'   : 28,
        'i16uOutputOffset'  : 30,
        'i16uConfigOffset'  : 32,
        'i16uFirstEntry'    : 34,
        'i16uEntries'       : 36,
        'i8uModuleState'    : 38,
        'i8uActive'         : 39,
        'i8uReserve'        : 40
}

dPrefix2Size = {
	'i8u'  : 'B',   # 1 byte, unsigned char
	'i16u' : 'H',	# 2 byte, unsigned short
	'i32u' : 'I'	# 4 byte, unsigned integer
}

def vprint(str):
    if bVerbose:
        print(str)

def dprint(str):
    if bDebug:
        print("DEBUG: " + str)
		
def mysyslog(str):
	if bSyslog:
		syslog.syslog(str)

def GetPictoryModules(f):
	bufsize_bytearray = size_SDeviceInfo * MAX_MODULES

	prm = (b'K'[0] << 8) + KB_GET_DEVICE_INFO_LIST   # the IOCTL-parameter is calculated using the ASCIIC 'K' which is shifted by 8 bits plus the ID of the desired function 
	devList = bytearray(bufsize_bytearray)
	# get list of devices
	ret = fcntl.ioctl(f, prm, devList)			  # The result values are written into a byte-array
	
	aModules = []
	if ret > 0:
		for i in range(ret):
			dictModule = {}
			for varInStruct, offsetInStruct in dOffsets_SDeviceInfo.items():
				strFormatFount = ''
				for varPrefix, strFmtSearch in dPrefix2Size.items():
					if varPrefix in varInStruct:
						strFormatFount = strFmtSearch
						break
				
				if '' == strFormatFount:
					print("Error: Variable prefix format not defined")
					sys.exit(1)

				offsetInByteArray = size_SDeviceInfo * i + offsetInStruct
				value = struct.unpack_from(strFormatFount, devList, offsetInByteArray )[0]
				dictModule[varInStruct] =  value			
			aModules.append(dictModule)

	return aModules

def GetTimer(ioData, timerID):
    if timerID > (TIMER_CNT - 1):
        print("Error: timerID")
    else:
        startPos = timerID * TIMER_SIZE
        endPos = startPos + TIMER_SIZE
        timerData = ioData[startPos: endPos]
        (on_day, on_hour, on_minute, off_day, off_hour, off_minute) = struct.unpack(TIMER_FMT, timerData)

        timerDict = {
            "OnDay" : on_day,
            "OnHour" 	: on_hour,
            "OnMinute" 	: on_minute,
            "OffDay" 	: off_day,
            "OffHour" 	: off_hour,
            "OffMinute" : off_minute
        }
        return timerDict

def GetStatusBytes(inData):
    (statusBytes) = struct.unpack(STATUS_FMT, inData)
    return statusBytes[0]		
		
def CalcStatus(inData, StatusBytes):
    dt_now = datetime.datetime.now()

    nowWeekday = dt_now.weekday()		# day Monday is 0 and Sunday is 6
    nowHour    = dt_now.hour
    nowMinute  = dt_now.minute

    flag_nowWeekday = 1 << nowWeekday

    for timerID in range(0, TIMER_CNT):
        dictTimer = GetTimer(inData, timerID)
        dprint('Timer #:{}'.format(timerID))
        # check on
        if dictTimer["OnDay"] & 1 << 7:
            dprint("\t On Active ok")
            if flag_nowWeekday & dictTimer["OnDay"]:
                dprint("\t On Day ok")
                if dictTimer["OnHour"] == nowHour:
                    dprint("\t On Hour ok")
                    if dictTimer["OnMinute"] == nowMinute:
                        dprint("\t On Minute ok")
                        StatusBytes |= 1 << timerID	# switch timer on
                        dprint("Switching Timer {0} : ON".format(timerID))
                        mysyslog("Switching Timer {0} : ON".format(timerID))
                    else:
                        dprint("\t On Minute NOK")
                else:
                    dprint("\t On Hour NOK")
            else:
                dprint("\t On Day NOK")
        else:
            dprint("\t On Active NOK")

        # check OFF
        if dictTimer["OffDay"] & 1 << 7:
            dprint("\t Off Active ok")
            if flag_nowWeekday & dictTimer["OffDay"]:
                dprint("\t Off Day ok")
                if dictTimer["OffHour"] == nowHour:
                    dprint("\t Off Hour ok")
                    if dictTimer["OffMinute"] == nowMinute:
                        dprint("\t Off Minute ok")
                        StatusBytes &= ~(1 << timerID)	# switch timer off
                        dprint("Switching Timer {0} : OFF".format(timerID))
                        mysyslog("Switching Timer {0} : OFF".format(timerID))
                    else:
                        dprint("\t Off Minute NOK")
                else:
                    dprint("\t Off Hour NOK")
            else:
                dprint("\t Off Day NOK")
        else:
            dprint("\t Off Active NOK")

    dprint("return StatusBytes : 0x{0:04X}".format(StatusBytes))
    return StatusBytes
		
def CalcAndUpdateStatus():
	# first the driver has to be opened by the "open" statement:
	f = open("/dev/piControl0","wb+",0)

	# get array of modules
	aModules  = GetPictoryModules(f)

	dprint("DEBUG: modules found : {0} ".format(len(aModules)))

	# find timer modules
	aTimerModules = []
	for dModule in aModules:
		if MODULETYPE_TIMER == dModule['i16uModuleType']:
			aTimerModules.append(dModule)

	vprint("timer modules found : {0} ".format(len(aTimerModules)))

	for dModule in aTimerModules:
		vprint("processing timer module at address: {0}".format(dModule['i8uAddress']))
		dprint("address: {0}, input offset: {1}, input length {2}, output offset: {3}, output length: {4}".format(
		dModule['i8uAddress'], dModule['i16uInputOffset'], dModule['i16uInputLength'], dModule['i16uOutputOffset'], dModule['i16uOutputLength'] ))

		# read Input and Output Data
		f.seek(dModule['i16uInputOffset'])
		inData = f.read(dModule['i16uInputLength'])

		f.seek(dModule['i16uOutputOffset'])
		outData = f.read(dModule['i16uOutputLength'])
		StatusBytes1 = GetStatusBytes(inData)
		StatusBytes2 = CalcStatus(outData, StatusBytes1)
		f.seek(dModule['i16uInputOffset'])
		f.write((StatusBytes2).to_bytes(2, byteorder='little', signed=False))
		
		vprint("Status Old: 0x{0:04X}, New: 0x{1:04X}".format(StatusBytes1, StatusBytes2))

	f.close()

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description='RevolutionPi Timer')
    parser.add_argument('--verbose', help='increase verbosity', action='store_true')
    parser.add_argument('--debug', help='print out debug messages', action='store_true')
    parser.add_argument('--syslog', help='log to syslog', action='store_true')
	
    args = parser.parse_args()

    if args.verbose:
        bVerbose = True

    if args.debug:
        bDebug = True

    if args.syslog:
        bSyslog = True

    CalcAndUpdateStatus()

